//----------------------------------------------------------------------------
//
// "Creature Box" -- flocking app
//
// by Christopher Rasmussen, cer@cis.udel.edu
//
// 1.0: initial version, March, 2014
// 1.1: updated for OpenGL 3.3, March 2016
//
//----------------------------------------------------------------------------

// Include standard headers

#include <stdio.h>
#include <stdlib.h>

// Include GLEW

#include <GL/glew.h>

// Include GLFW

#include <glfw3.h>

// Include GLM

#include <glm/glm.hpp>
#include <glm/gtx/transform.hpp>
#include <glm/gtc/matrix_transform.hpp>
using namespace glm;

// for loading GLSL shaders

#include <common/shader.hpp>

// creature-specific stuff

#include "Flocker.hh"
#include "Predator.hh"

//----------------------------------------------------------------------------


// to avoid gimbal lock issues...

#define MAX_LATITUDE_DEGS     89.0
#define MIN_LATITUDE_DEGS    -89.0

#define BOX_WIDTH  9.0f
#define BOX_HEIGHT 5.0f
#define BOX_DEPTH  7.0f

#define CAMERA_MODE_ORBIT    1
#define CAMERA_MODE_CREATURE 2

#define MIN_ORBIT_CAM_RADIUS    (BOX_DEPTH)
#define MAX_ORBIT_CAM_RADIUS    (25.0)

#define DEFAULT_ORBIT_CAM_RADIUS            22.5
#define DEFAULT_ORBIT_CAM_LATITUDE_DEGS     0.0
#define DEFAULT_ORBIT_CAM_LONGITUDE_DEGS    90.0

//----------------------------------------------------------------------------

// some convenient globals 

GLFWwindow* window;

GLuint programID;
GLuint MatrixID;
GLuint VertexArrayID;

// these along with Model matrix make MVP transform

glm::mat4 Projection;
glm::mat4 View;

// sim-related globals

bool is_paused = false;

double orbit_cam_radius = DEFAULT_ORBIT_CAM_RADIUS;
double orbit_cam_delta_radius = 0.1;

double orbit_cam_latitude_degs = DEFAULT_ORBIT_CAM_LATITUDE_DEGS;
double orbit_cam_longitude_degs = DEFAULT_ORBIT_CAM_LONGITUDE_DEGS;
double orbit_cam_delta_theta_degs = 1.0;

int win_scale_factor = 60;
int win_w = win_scale_factor * 16.0;
int win_h = win_scale_factor * 9.0;
 
int camera_mode = CAMERA_MODE_ORBIT;

int num_flockers = 50;   // 400 is "comfortable" max on my machine

extern int flocker_history_length;
extern int flocker_draw_mode;
extern vector <Flocker *> flocker_array;    
extern vector <vector <double> > flocker_squared_distance;

GLuint box_vertexbuffer;
GLuint box_colorbuffer;

//----------------------------------------------------------------------------

void end_program()
{
  // Cleanup VBOs and shader

  glDeleteBuffers(1, &box_vertexbuffer);
  glDeleteBuffers(1, &box_colorbuffer);
  glDeleteProgram(programID);
  glDeleteVertexArrays(1, &VertexArrayID);
  
  // Close OpenGL window and terminate GLFW

  glfwTerminate();

  exit(1);
}

//----------------------------------------------------------------------------

// corner "origin" is at (0, 0, 0) -- must translate to center

void draw_box(glm::mat4 Model)
{
  // Our ModelViewProjection : multiplication of our 3 matrices

  glm::mat4 MVP = Projection * View * Model;

  // make this transform available to shaders  

  glUniformMatrix4fv(MatrixID, 1, GL_FALSE, &MVP[0][0]);

  // 1st attribute buffer : vertices

  glEnableVertexAttribArray(0);
  glBindBuffer(GL_ARRAY_BUFFER, box_vertexbuffer);
  glVertexAttribPointer(0,                  // attribute. 0 to match the layout in the shader.
			3,                  // size
			GL_FLOAT,           // type
			GL_FALSE,           // normalized?
			0,                  // stride
			(void*)0            // array buffer offset
			);
  
  // 2nd attribute buffer : colors

  glEnableVertexAttribArray(1);
  glBindBuffer(GL_ARRAY_BUFFER, box_colorbuffer);
  glVertexAttribPointer(1,                                // attribute. 1 to match the layout in the shader.
			3,                                // size
			GL_FLOAT,                         // type
			GL_FALSE,                         // normalized?
			0,                                // stride
			(void*)0                          // array buffer offset
			);

  // Draw the box!

  glDrawArrays(GL_LINES, 0, 24); 
  
  glDisableVertexAttribArray(0);
  glDisableVertexAttribArray(1);
}

//----------------------------------------------------------------------------

// handle key presses

void key_callback(GLFWwindow* window, int key, int scancode, int action, int mods)
{
  // quit

  if (key == GLFW_KEY_Q && action == GLFW_PRESS)
    end_program();

  // pause

  else if (key == GLFW_KEY_SPACE && action == GLFW_PRESS)
    is_paused = !is_paused;

  // orbit rotate

  else if (key == GLFW_KEY_A && (action == GLFW_PRESS || action == GLFW_REPEAT)) 
    orbit_cam_longitude_degs -= orbit_cam_delta_theta_degs;
  else if (key == GLFW_KEY_D && (action == GLFW_PRESS || action == GLFW_REPEAT))
    orbit_cam_longitude_degs += orbit_cam_delta_theta_degs;
  else if (key == GLFW_KEY_W && (action == GLFW_PRESS || action == GLFW_REPEAT)) {
    if (orbit_cam_latitude_degs + orbit_cam_delta_theta_degs <= MAX_LATITUDE_DEGS)
      orbit_cam_latitude_degs += orbit_cam_delta_theta_degs;
  }
  else if (key == GLFW_KEY_S && (action == GLFW_PRESS || action == GLFW_REPEAT)) {
    if (orbit_cam_latitude_degs - orbit_cam_delta_theta_degs >= MIN_LATITUDE_DEGS)
      orbit_cam_latitude_degs -= orbit_cam_delta_theta_degs;
  }

  // orbit zoom in/out

  else if (key == GLFW_KEY_Z && (action == GLFW_PRESS || action == GLFW_REPEAT)) {
    if (orbit_cam_radius + orbit_cam_delta_radius <= MAX_ORBIT_CAM_RADIUS)
      orbit_cam_radius += orbit_cam_delta_radius;
  }
  else if (key == GLFW_KEY_C && (action == GLFW_PRESS || action == GLFW_REPEAT)) {
    if (orbit_cam_radius - orbit_cam_delta_radius >= MIN_ORBIT_CAM_RADIUS)
      orbit_cam_radius -= orbit_cam_delta_radius;
  }

  // orbit pose reset

  else if (key == GLFW_KEY_X && action == GLFW_PRESS) {
    orbit_cam_radius = DEFAULT_ORBIT_CAM_RADIUS;
    orbit_cam_latitude_degs = DEFAULT_ORBIT_CAM_LATITUDE_DEGS;
    orbit_cam_longitude_degs = DEFAULT_ORBIT_CAM_LONGITUDE_DEGS;
  }

  // flocker drawing options

  else if (key == GLFW_KEY_8 && action == GLFW_PRESS)
    flocker_draw_mode = DRAW_MODE_POLY;
  else if (key == GLFW_KEY_9 && action == GLFW_PRESS)
    flocker_draw_mode = DRAW_MODE_AXES;
  else if (key == GLFW_KEY_0 && action == GLFW_PRESS)
    flocker_draw_mode = DRAW_MODE_HISTORY;
}

//----------------------------------------------------------------------------

// allocate simulation data structures and populate them

void initialize_simulation()
{
  // box geometry with corner at origin

  static const GLfloat box_vertex_buffer_data[] = {
    0.0f, 0.0f, 0.0f,                      // X axis
    BOX_WIDTH, 0.0f, 0.0f,
    0.0f, 0.0f, 0.0f,                      // Y axis
    0.0f, BOX_HEIGHT, 0.0f,
    0.0f, 0.0f, 0.0f,                      // Z axis
    0.0f, 0.0f, BOX_DEPTH,
    BOX_WIDTH, BOX_HEIGHT, BOX_DEPTH,      // other edges
    0.0f, BOX_HEIGHT, BOX_DEPTH,
    BOX_WIDTH, BOX_HEIGHT, 0.0f,
    0.0f, BOX_HEIGHT, 0.0f,
    BOX_WIDTH, 0.0f, BOX_DEPTH,
    0.0f, 0.0f, BOX_DEPTH,
    BOX_WIDTH, BOX_HEIGHT, BOX_DEPTH,
    BOX_WIDTH, 0.0f, BOX_DEPTH,
    0.0f, BOX_HEIGHT, BOX_DEPTH,
    0.0f, 0.0f, BOX_DEPTH,
    BOX_WIDTH, BOX_HEIGHT, 0.0f,
    BOX_WIDTH, 0.0f, 0.0f,
    BOX_WIDTH, BOX_HEIGHT, BOX_DEPTH,
    BOX_WIDTH, BOX_HEIGHT, 0.0f,
    0.0f, BOX_HEIGHT, BOX_DEPTH,
    0.0f, BOX_HEIGHT, 0.0f,
    BOX_WIDTH, 0.0f, BOX_DEPTH,
    BOX_WIDTH, 0.0f, 0.0f,
  };

  glGenBuffers(1, &box_vertexbuffer);
  glBindBuffer(GL_ARRAY_BUFFER, box_vertexbuffer);
  glBufferData(GL_ARRAY_BUFFER, sizeof(box_vertex_buffer_data), box_vertex_buffer_data, GL_STATIC_DRAW);

  // "axis" edges are colored, rest are gray

  static const GLfloat box_color_buffer_data[] = { 
    1.0f, 0.0f, 0.0f,       // X axis is red 
    1.0f, 0.0f, 0.0f,        
    0.0f, 1.0f, 0.0f,       // Y axis is green 
    0.0f, 1.0f, 0.0f,
    0.0f, 0.0f, 1.0f,       // Z axis is blue
    0.0f, 0.0f, 1.0f,
    0.5f, 0.5f, 0.5f,       // all other edges gray
    0.5f, 0.5f, 0.5f,
    0.5f, 0.5f, 0.5f,
    0.5f, 0.5f, 0.5f,
    0.5f, 0.5f, 0.5f,
    0.5f, 0.5f, 0.5f,
    0.5f, 0.5f, 0.5f,
    0.5f, 0.5f, 0.5f,
    0.5f, 0.5f, 0.5f,
    0.5f, 0.5f, 0.5f,
    0.5f, 0.5f, 0.5f,
    0.5f, 0.5f, 0.5f,
    0.5f, 0.5f, 0.5f,
    0.5f, 0.5f, 0.5f,
    0.5f, 0.5f, 0.5f,
    0.5f, 0.5f, 0.5f,
    0.5f, 0.5f, 0.5f,
    0.5f, 0.5f, 0.5f,
  };

  glGenBuffers(1, &box_colorbuffer);
  glBindBuffer(GL_ARRAY_BUFFER, box_colorbuffer);
  glBufferData(GL_ARRAY_BUFFER, sizeof(box_color_buffer_data), box_color_buffer_data, GL_STATIC_DRAW);
  
  // the simulation proper

  initialize_random();

  flocker_squared_distance.resize(num_flockers);
  flocker_array.clear();

  for (int i = 0; i < num_flockers; i++) {
    flocker_array.push_back(new Flocker(i, 
					  uniform_random(0, BOX_WIDTH), uniform_random(0, BOX_HEIGHT), uniform_random(0, BOX_DEPTH),
					  uniform_random(-0.01, 0.01), uniform_random(-0.01, 0.01), uniform_random(-0.01, 0.01),
					  0.002,            // randomness
					  0.05, 0.5, 0.02,  // min, max separation distance, weight
					  0.5,  1.0, 0.001, // min, max alignment distance, weight
					  1.0,  1.5, 0.001, // min, max cohesion distance, weight
					  1.0,  1.0, 1.0,
					  flocker_history_length));

    flocker_squared_distance[i].resize(num_flockers);
  }
}

//----------------------------------------------------------------------------

// move creatures around -- no drawing

void update_simulation()
{
  int i;

  // precalculate inter-flocker distances

  calculate_flocker_squared_distances();

  // get new_position, new_velocity for each flocker

  for (i = 0; i < flocker_array.size(); i++) 
    flocker_array[i]->update();

  // handle wrapping and make new position, velocity into current

  for (i = 0; i < flocker_array.size(); i++) 
    flocker_array[i]->finalize_update(BOX_WIDTH, BOX_HEIGHT, BOX_DEPTH);
}

//----------------------------------------------------------------------------

// place the camera here

void setup_camera()
{
  Projection = glm::perspective(50.0f, (float) win_w / (float) win_h, 0.1f, 35.0f);

  if (camera_mode == CAMERA_MODE_ORBIT) {
    
    double orbit_cam_azimuth = glm::radians(orbit_cam_longitude_degs);
    double orbit_cam_inclination = glm::radians(90.0 - orbit_cam_latitude_degs);
    
    double x_cam = orbit_cam_radius * sin(orbit_cam_inclination) * cos(orbit_cam_azimuth); // 0.5 * BOX_WIDTH;
    double z_cam = orbit_cam_radius * sin(orbit_cam_inclination) * sin(orbit_cam_azimuth); // 0.5 * BOX_HEIGHT;
    double y_cam = orbit_cam_radius * cos(orbit_cam_inclination); // 15.0;
    
    View = glm::lookAt(glm::vec3(x_cam, y_cam, z_cam),   // Camera location in World Space
		       glm::vec3(0,0,0),                 // and looks at the origin
		       glm::vec3(0,-1,0)                  // Head is up (set to 0,-1,0 to look upside-down)
		       );
  }
  else {
    
    printf("only orbit camera mode currently supported\n");
    exit(1);
    
  }
}

//----------------------------------------------------------------------------

int main( void )
{
  // Initialise GLFW

  if( !glfwInit() )
    {
      fprintf( stderr, "Failed to initialize GLFW\n" );
      getchar();
      return -1;
    }
  
  glfwWindowHint(GLFW_SAMPLES, 4);
  glfwWindowHint(GLFW_CONTEXT_VERSION_MAJOR, 3);
  glfwWindowHint(GLFW_CONTEXT_VERSION_MINOR, 3);
  glfwWindowHint(GLFW_OPENGL_FORWARD_COMPAT, GL_TRUE); // To make MacOS happy; should not be needed
  glfwWindowHint(GLFW_OPENGL_PROFILE, GLFW_OPENGL_CORE_PROFILE); //We don't want the old OpenGL 
  
  // Open a window and create its OpenGL context

  window = glfwCreateWindow(win_w, win_h, "creatures", NULL, NULL);
  if( window == NULL ){
    fprintf( stderr, "Failed to open GLFW window. If you have an Intel GPU, they are not 3.3 compatible. Try the 2.1 version of the tutorials.\n" );
    getchar();
    glfwTerminate();
    return -1;
  }
  glfwMakeContextCurrent(window);
  
  // Initialize GLEW

  glewExperimental = true; // Needed for core profile
  if (glewInit() != GLEW_OK) {
    fprintf(stderr, "Failed to initialize GLEW\n");
    getchar();
    glfwTerminate();
    return -1;
  }
  
  // simulation

  initialize_simulation();

  // register all callbacks

  glfwSetKeyCallback(window, key_callback);

  // background color, etc.

  glClearColor(0.0f, 0.0f, 0.1f, 0.0f);
  glEnable(GL_DEPTH_TEST);

  glGenVertexArrays(1, &VertexArrayID);
  glBindVertexArray(VertexArrayID);
  
  // Create and compile our GLSL program from the shaders

  programID = LoadShaders( "Creatures.vertexshader", "Creatures.fragmentshader" );
  
  // Get a handle for our "MVP" uniform

  MatrixID = glGetUniformLocation(programID, "MVP");
  
    // Use our shader

  glUseProgram(programID);

  // enter simulate-render loop (with event handling)
  
  do {

    // STEP THE SIMULATION

    if (!is_paused)
      update_simulation();

    // RENDER IT

    // Clear the screen

    glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
        
    // set model transform and color for each triangle and draw it offscreen

    setup_camera();

    // a centering translation for viewing

    glm::mat4 M = glm::translate(glm::vec3(-0.5f * BOX_WIDTH, -0.5f * BOX_HEIGHT, -0.5f * BOX_DEPTH));

    // draw box and creatures

    draw_box(M);

    for (int i = 0; i < flocker_array.size(); i++) 
      flocker_array[i]->draw(M);

    // Swap buffers

    glfwSwapBuffers(window);
    glfwPollEvents();

  } while ( glfwWindowShouldClose(window) == 0 );
  
  end_program();
  
  return 0;
}

//----------------------------------------------------------------------------
//----------------------------------------------------------------------------
